import cx_Oracle
import json
import logging
from configparser import ConfigParser
from datetime import datetime, date
import re
import sys
import os

config = ConfigParser()
config.read(os.path.join(os.path.dirname(__file__), 'config.ini'))

cx_Oracle.init_oracle_client(lib_dir=config['PATHS']['instantclient_dir'])

log_dir = config['LOGGING']['log_dir']
log_file_path = os.path.join(log_dir, f'Physical_query_response_load_to_database.log')
logging.basicConfig(filename=log_file_path, level=logging.INFO, format='%(asctime)s - %(levelname)s - %(message)s')

def parse_datetime_with_milliseconds(date_str):
    try:
        return datetime.strptime(date_str, "%Y-%m-%dT%H:%M:%S.%fZ")
    except ValueError:
        return datetime.strptime(date_str, "%Y-%m-%dT%H:%M:%SZ")

BATCH_SIZE = 1000

def fetch_last_load_ts(cursor):
    try:
        cursor.execute("Select max(log_ts) from PHYSICAL_QUERY_RESPONSE_DATA")
        last_load_ts = cursor.fetchone()[0]
        if last_load_ts is None:
            last_load_ts = datetime.strptime('2023-01-01 00:00:00.000', '%Y-%m-%d %H:%M:%S.%f')
        return last_load_ts
    except Exception as e:
        logging.error(f"Error fetching last job timestamp: {e}")
        return None

print(f"Physical Query Responses JSON data started loading into PHYSICAL_QUERY_RESPONSE_DATA. Please check the log {log_file_path}")

def batch_insert(cursor, records, start_index):
    try:
        cursor.executemany("""
            INSERT INTO PHYSICAL_QUERY_RESPONSE_DATA(ecid, total_time_sec,fetch_time_sec, execute_time_sec, query_id, physical_req_hash, user_name, log_content_id, instance_name, log_ts, log_dt)
            VALUES (:ecid, :total_time,:fetch_time, :execute_time, :query_id, :physical_request_hash, :user_id, :log_content_id, :instance_name, TO_TIMESTAMP(:log_ts, 'YYYY-MM-DD HH24:MI:SS.FF'), TO_DATE(:log_dt, 'DD/MM/YYYY'))
        """, records)
        logging.info(f"Inserted JSON rows {start_index} to {start_index + len(records) - 1} into the database.")
    except Exception as e:
        logging.error(f"Error inserting records {start_index} to {start_index + len(records) - 1}: {e}")

def load_json_to_database(connection, json_file_path):
    load_start_time = datetime.now().strftime("%d-%m-%Y %H:%M:%S.%f")

    # Fetch last load timestamp
    cursor = connection.cursor()
    last_load_ts = fetch_last_load_ts(cursor)
    cursor.close()
    
    with open(json_file_path) as json_file:
        data_list = json.load(json_file)

    logging.info(f"Number of records in JSON file: {len(data_list)}")

    cursor = connection.cursor()
    records = []
    total_rows_inserted = 0
    start_index = 1

    def process_message(message, data_content, log_content, log_time):
        fetch_time = execute_time = query_id = physical_request_hash = total_time = None
        for part in message.split(","):
            if "Fetch time" in part:
                fetch_time = float(part.split("Fetch time")[1].split()[0])
            elif "Execute time" in part:
                execute_time = float(part.split("Execute time")[1].split()[0])
            elif "Total time" in part:
                total_time = float(part.split("Total time")[1].split()[0])
            elif "id <<" in part:
                query_id = part.split("id <<")[1].split(">>")[0]
            elif "physical request hash" in part:
                physical_request_hash = re.search(r'physical request hash\s*(\S+)', part)
                if physical_request_hash:
                    physical_request_hash = physical_request_hash.group(1).strip()

        record = {
            'ecid': data_content.get('ecid'),
            'fetch_time': fetch_time,
            'execute_time': execute_time,
            'query_id': query_id,
            'physical_request_hash': physical_request_hash,
            'user_id': data_content.get('userId'),
            'log_content_id': log_content.get('id'),
            'instance_name': log_content.get('source'),
            'log_ts': log_time.strftime("%Y-%m-%d %H:%M:%S.%f")[:-3],
            'log_dt': log_time.date().strftime("%d/%m/%Y"),
            'total_time': total_time
        }

        return record

    for i, data in enumerate(data_list, start=1):
        log_content = data.get('logContent', {})
        data_content = log_content.get('data', {})
        oracle_details = log_content.get('oracle', {})

        log_time_str = log_content.get('time')
        log_time = parse_datetime_with_milliseconds(log_time_str)

        message = data_content.get('message')
        if "Physical query response" in message:
            record = process_message(message, data_content, log_content, log_time)
            records.append(record)

            if len(records) >= BATCH_SIZE or (i == len(data_list)):
                batch_insert(cursor, records, start_index)
                start_index += len(records)
                total_rows_inserted += len(records)
                records = []

    if records:
        batch_insert(cursor, records, start_index)
        total_rows_inserted += len(records)

    connection.commit()
    cursor.close()
    logging.info(f"OAC Physical Query Responses JSON data loaded successfully into PHYSICAL_QUERY_RESPONSE_DATA table. Total rows inserted: {total_rows_inserted}")
    print(f"OAC Physical Query Responses JSON data has been successfully loaded into the database.Total rows inserted: {total_rows_inserted}")
    logging.info(f"Deleting duplicate rows where log_ts between {last_load_ts} and {load_start_time}")

    try:
        cursor = connection.cursor()
        cursor.execute("""
            DELETE FROM PHYSICAL_QUERY_RESPONSE_DATA WHERE ROWID NOT IN (
            SELECT MIN(ROWID) FROM PHYSICAL_QUERY_RESPONSE_DATA WHERE log_ts BETWEEN TO_TIMESTAMP(:last_load_ts, 'YYYY-MM-DD HH24:MI:SS.FF') AND TO_TIMESTAMP(:load_start_time, 'DD-MM-YYYY HH24:MI:SS.FF')
            GROUP BY ecid,total_time_sec,fetch_time_sec,execute_time_sec,query_id,physical_req_hash,user_name,log_content_id,instance_name,log_ts,log_dt
           ) AND (log_ts BETWEEN TO_TIMESTAMP(:last_load_ts, 'YYYY-MM-DD HH24:MI:SS.FF') AND TO_TIMESTAMP(:load_start_time, 'DD-MM-YYYY HH24:MI:SS.FF')) """,{'load_start_time': load_start_time, 'last_load_ts': last_load_ts})
        deleted_rows = cursor.rowcount
        connection.commit()
        logging.info(f"{deleted_rows} duplicate rows deleted successfully.")
    except Exception as e:
        logging.error(f"Error deleting duplicate rows: {e}")
    finally:
        cursor.close()
        logging.basicConfig(filename='error.log', level=logging.ERROR, format='%(asctime)s - %(levelname)s - %(message)s')

if __name__ == "__main__":
    try:
        connection = cx_Oracle.connect(config['DATABASE']['username'], config['DATABASE']['password'], config['DATABASE']['service_name'])
        print("Database connection established successfully.")
        load_json_to_database(connection, config['DATA']['data_dir'] + f'/diagnostics_logs_json_data.json')
    except cx_Oracle.Error as oracle_error:
        logging.exception("An Oracle Database error occurred")
        print("Failed to load JSON data into the database due to Oracle Database error.")
        sys.exit(1)
    except Exception as generic_error:
        logging.exception("An error occurred")
        print("Failed to load JSON data into the database due to an error.")
        sys.exit(1)