import cx_Oracle
import json
import logging
import os
import sys
from datetime import datetime, date
from configparser import ConfigParser
import re

config = ConfigParser()
config.read(os.path.join(os.path.dirname(__file__), 'config.ini'))

# Set the Oracle client library path
cx_Oracle.init_oracle_client(lib_dir=config['PATHS']['instantclient_dir'])

# Configuring logging
log_dir = config['LOGGING']['log_dir']
log_file_path = os.path.join(log_dir, 'initblocks_json_to_database_load.log')
logging.basicConfig(filename=log_file_path, level=logging.DEBUG, format='%(asctime)s - %(levelname)s - %(message)s')

print(f"Loading Init blocks JSON data into the database. Logging into file: {log_file_path}")
logging.info("Init blocks JSON data loading process started...")

def parse_datetime_with_milliseconds(date_str):
    try:
        return datetime.strptime(date_str, "%Y-%m-%dT%H:%M:%S.%fZ")
    except ValueError:
        return datetime.strptime(date_str, "%Y-%m-%dT%H:%M:%SZ")

BATCH_SIZE = 1000

def fetch_last_load_ts(cursor):
    try:
        cursor.execute("SELECT MAX(log_ts) FROM INITBLOCK_LOG_DATA")
        last_load_ts = cursor.fetchone()[0]
        if last_load_ts is None:
            last_load_ts = datetime.strptime('2023-01-01 00:00:00.000', '%Y-%m-%d %H:%M:%S.%f')
        return last_load_ts
    except Exception as e:
        logging.error(f"Error fetching last job timestamp: {e}")
        return None

def batch_insert(cursor, records):
    cursor.executemany("""
        INSERT INTO INITBLOCK_LOG_DATA (ecid,category,user_name,log_content_id,instance_name,log_ts,log_dt,message,type,sql_query,status,init_block_name)
        VALUES (:ecid,:category,:user_id,:id,:source,TO_TIMESTAMP(:log_ts, 'YYYY-MM-DD HH24:MI:SS.FF'), TO_DATE(:log_dt, 'DD/MM/YYYY'),:message,:init_block_type,:sql_query,:status,:init_block_name)
    """, records)

def load_json_to_database(connection, json_file_path):
    load_start_time = datetime.now().strftime("%d-%m-%Y %H:%M:%S.%f")

    # Fetch last load timestamp
    cursor = connection.cursor()
    last_load_ts = fetch_last_load_ts(cursor)
    cursor.close()
    
    with open(json_file_path) as json_file:
        data_list = json.load(json_file)

    logging.info(f"Number of records in JSON file: {len(data_list)}")
    cursor = connection.cursor()
    records = []
    total_rows_inserted = 0

    for i, data in enumerate(data_list, start=1):
        log_content = data.get('logContent', {})
        data_content = log_content.get('data', {})
        oracle_details = log_content.get('oracle', {})

        log_time_str = log_content.get('time')
        log_time = parse_datetime_with_milliseconds(log_time_str)

        ingested_time_str = oracle_details.get('ingestedtime')
        ingested_time = parse_datetime_with_milliseconds(ingested_time_str)

        message = data_content.get('message')
        
        # Extracting init_block_name from message
        if "Init Block Execution Info" in message:
            if "init block" in message:
                if "has errors" in message:
                    init_block_match = re.search(r"init\sblock\s(.*?)(?:\s*has\ errors|$)", message)
                    if init_block_match:
                        init_block_name = init_block_match.group(1)
                else:
                    init_block_match = re.search(r"init\sblock\s(?:'([^']*)'|([^']*))", message)
                    if init_block_match:
                        init_block_name = init_block_match.group(1) or init_block_match.group(2)
            else:
                init_block_match = re.search(r"(.*?)\s*executed\ssuccessfully", message)
                if init_block_match:
                    init_block_name = init_block_match.group(1).strip()

            # Extracting the Init block type and status from message
            if "Init Block" in message or "init block" in message:
                if "executed successfully" in message:
                    status = 'Completed'
                    if "Repository init block" in message:
                        init_block_type = "Repository"
                    else:
                        init_block_type = "Session"
                elif "has errors" in message:
                    status = 'Failed'
                    init_block_type = None
                elif "Executing session init block" in message:
                    init_block_type = "Session"
                    status = 'Running'
                elif "Executing repository init block" in message:
                    init_block_type = "Repository"
                    status = 'Running'
                elif "Skipping execution of Initialization Block" in message:
                    status = 'Skipped'
                    init_block_type = None
                elif "Executing init block" in message:
                    status = 'Running'
                    init_block_type = "Session"  # Assuming it's a session block
                else:
                    init_block_type = None
                    status = None

                # Extracting the Init block sql from message to insert in sql_query column
                sql_query_match = re.search(r"initializer string: (.+)$", message, re.DOTALL)
                sql_query = sql_query_match.group(1)[:4000] if sql_query_match else None
            else:
                status = None
                init_block_type = None
                sql_query = None
                init_block_name = None
            
            record = {
                'ecid': data_content.get('ecid'),
                'category': data_content.get('category'),
                'user_id': data_content.get('userId'),
                'id': log_content.get('id'),
                'source': log_content.get('source'),
                'log_ts': log_time.strftime("%Y-%m-%d %H:%M:%S.%f")[:-3],
                'log_dt': log_time.date().strftime("%d/%m/%Y"),
                'message': message,
                'init_block_type': init_block_type,
                'sql_query': sql_query,
                'status': status,
                'init_block_name': init_block_name
            }
            records.append(record)

            if len(records) >= BATCH_SIZE or i == len(data_list):
                batch_insert(cursor, records)
                total_rows_inserted += len(records)
                logging.info(f"Inserted JSON row {i} into the database.")
                records = []

    connection.commit()
    cursor.close()
    logging.info(f"Init blocks JSON data loaded successfully into INITBLOCK_LOG_DATA table. Total rows inserted: {total_rows_inserted}")
    print(f"Init blocks JSON data has been successfully loaded into the database.{total_rows_inserted}")
    logging.info(f"Deleting duplicate rows where log_ts between {last_load_ts} and {load_start_time}")

    try:
        cursor = connection.cursor()
        cursor.execute("""
            DELETE FROM INITBLOCK_LOG_DATA WHERE ROWID NOT IN (
            SELECT MIN(ROWID) FROM INITBLOCK_LOG_DATA WHERE log_ts BETWEEN TO_TIMESTAMP(:last_load_ts, 'YYYY-MM-DD HH24:MI:SS.FF') AND TO_TIMESTAMP(:load_start_time, 'DD-MM-YYYY HH24:MI:SS.FF')
            GROUP BY ecid,category,user_name,log_content_id,type,init_block_name,status,instance_name,log_ts,log_dt
            ) AND (log_ts BETWEEN TO_TIMESTAMP(:last_load_ts, 'YYYY-MM-DD HH24:MI:SS.FF') AND TO_TIMESTAMP(:load_start_time, 'DD-MM-YYYY HH24:MI:SS.FF'))""",{'load_start_time': load_start_time, 'last_load_ts': last_load_ts})
        deleted_rows = cursor.rowcount
        connection.commit()
        logging.info(f"{deleted_rows} duplicate rows deleted successfully.")
    except Exception as e:
        logging.error(f"Error deleting duplicate rows: {e}")
    finally:
        cursor.close()
        logging.basicConfig(filename='error.log', level=logging.ERROR, format='%(asctime)s - %(levelname)s - %(message)s')

if __name__ == "__main__":
    try:
        connection = cx_Oracle.connect(config['DATABASE']['username'], config['DATABASE']['password'], config['DATABASE']['service_name'])
        print("Database connection established successfully.")      
        load_json_to_database(connection, os.path.join(config['DATA']['data_dir'], 'init_blocks_logs_json_data.json'))

    except cx_Oracle.DatabaseError as db_error:
        print("Failed to connect to the database or execute a query. Please check the log file for details.")
        logging.exception("Database connection or query execution error occurred")
        sys.exit(1)

    except Exception as e:
        logging.exception("An unexpected error occurred")
        print("An unexpected error occurred. Please check the log file for details.")
        sys.exit(1)