import logging
import datetime
import json
import os
from configparser import ConfigParser
from datetime import date, datetime, timedelta
import oci

today = date.today()
Date = today

config_parser = ConfigParser()
config_parser.read(os.path.join(os.path.dirname(__file__), 'config.ini'))

compartment_ocid = config_parser['OCI']['compartment_ocid']
log_group_ocid = config_parser['OCI']['log_group_ocid']
log_ocid = config_parser['OCI']['audit_log_ocid']


search_query = f"search \"{compartment_ocid}/{log_group_ocid}/{log_ocid}\" | sort by datetime desc"

# Read load type from config.ini
load_type = config_parser['LOAD'].get('load_type').lower()
if load_type not in ['full_load', 'incremental']:
    raise ValueError("Invalid load type specified in config.ini. Supported types are 'full_load' and 'incremental'.")

# Read the full load parameters from config.ini
if load_type == 'full_load':
    start_time_year = int(config_parser['FULL_LOAD']['start_year'])
    start_time_month = int(config_parser['FULL_LOAD']['start_month'])
    start_time_day = int(config_parser['FULL_LOAD']['start_day'])
    end_time_year = int(config_parser['FULL_LOAD']['end_year'])
    end_time_month = int(config_parser['FULL_LOAD']['end_month'])
    end_time_day = int(config_parser['FULL_LOAD']['end_day'])
    start_date_time = datetime(year=start_time_year, month=start_time_month, day=start_time_day, hour=00, minute=00, second=00, microsecond=000)
    end_date_time = datetime(year=end_time_year, month=end_time_month, day=end_time_day, hour=00, minute=00, second=00, microsecond=000)
    load_interval = int(config_parser['FULL_LOAD'].get('load_interval_minutes'))
elif load_type == 'incremental':
    load_interval = int(config_parser['INCREMENTAL'].get('load_interval_minutes'))
    current_time_utc = datetime.utcnow().replace(second=0, microsecond=0)
    end_date_time = current_time_utc
    start_date_time = end_date_time - timedelta(minutes=15)

# Configuring the logging
log_dir = config_parser['LOGGING']['log_dir']
log_file_path = os.path.join(log_dir, f'audit_logs_json_data_download.log')
logging.basicConfig(filename=log_file_path, level=logging.INFO, format='%(asctime)s - %(levelname)s - %(message)s')

print("OAC Audit JSON data download started from OCI Logging Service. Logging into log file %s" % log_file_path)
logging.info("OAC Audit logs data download started. Please wait...")

# Define console handler with desired format and level
console = logging.StreamHandler()
console.setLevel(logging.ERROR)
formatter = logging.Formatter('%(asctime)s - %(levelname)s - %(message)s')
console.setFormatter(formatter)
logging.getLogger('').addHandler(console)
config = oci.config.from_file(profile_name='DEFAULT')
service_endpoint = 'https://logging.us-ashburn-1.oci.oraclecloud.com'
client = oci.loggingsearch.LogSearchClient(config, service_endpoint=service_endpoint)

def download_oci_logs(query, start_date_time, end_date_time):
    loop_start_ts = start_date_time
    loop_end_ts = loop_start_ts + timedelta(minutes=load_interval)

    if end_date_time < loop_end_ts:
        loop_end_ts = end_date_time

    out_json_file_full_path = os.path.join(config_parser['DATA']['data_dir'], f'audit_logs_json_data.json')
    out_file = open(out_json_file_full_path, "a+")
    out_file.write("[")
    is_first_line = True
    total_rows = 0

    while True:
        log_line_array = paginated_log_search(query, loop_start_ts, loop_end_ts)
        total_rows += len(log_line_array)

        for log_line in log_line_array:
            if is_first_line:
                prefix = ""
            else:
                prefix = "\n,"
            out_file.write(prefix + log_line)
            is_first_line = False

        logging.info("Download of logs from %s to %s is completed" % (loop_start_ts.strftime('%Y-%m-%d %H:%M:%S.%f'), loop_end_ts.strftime('%Y-%m-%d %H:%M:%S.%f')))

        loop_start_ts = loop_end_ts + timedelta(milliseconds=1) 
        if loop_start_ts > end_date_time:
            break

        loop_end_ts = loop_start_ts + timedelta(minutes=load_interval)
        if end_date_time < loop_end_ts:
            loop_end_ts = end_date_time

    out_file.write("]")
    out_file.close()
    return total_rows

def paginated_log_search(query, start_date_time, end_date_time):
    page = None
    log_line_array = []
    while True:
        details = oci.loggingsearch.models.SearchLogsDetails(
            time_start=start_date_time,
            time_end=end_date_time,
            search_query=query,
            is_return_field_info=False
        )
        response = client.search_logs(search_logs_details=details, limit=1000, page=page)

        for log_entry in response.data.results:
            log_line_array.append(json.dumps(log_entry.data))

        if response.has_next_page:
            page = response.next_page
        else:
            break

    return log_line_array
try:
    total_rows = download_oci_logs(search_query, start_date_time, end_date_time)
    print("OAC Audit JSON data download successfully completed. Total rows downloaded: %d" % total_rows)
    logging.info("OAC Audit JSON data download successfully completed: %d" % total_rows)
except Exception as e:
    # Log error message and exit with error
    print("Downloading Audit log data from OCI logs failed. Please check the log %s for details." % log_file_path)
    logging.error("Downloading Audit log data from OCI logs failed: %s" % str(e))
    raise SystemExit(1)