import os
import subprocess
import configparser
import shutil
import logging
import concurrent.futures
import glob
from datetime import datetime

config = configparser.ConfigParser()
config.read(os.path.join(os.path.dirname(__file__), 'config.ini'))

# Configuring the logging
log_file_path = os.path.join(config['LOGGING']['log_dir'], 'data_load_status.log')
logging.basicConfig(filename=log_file_path, level=logging.INFO, format='%(asctime)s - %(levelname)s - %(message)s')
logging.info("********************************Data loads refresh has been started****************************************")
print(f"Data loads refresh started. Check the log file for load status: {log_file_path}")

export_scripts = [
    'export_dv_export_logs_to_json.py',
    'export_audit_logs_to_json.py',
    'export_dv_job_logs_to_json.py',
    'export_init_logs_to_json.py',
    'export_diagnostics_logs_to_json.py'
]

import_scripts = {
    'import_audit_logs_data_load.py': 'audit_logs_json_data.json',
    'import_dv_jobs_json_data_load.py': 'dv_jobs_logs_json_data.json',
    'import_dataflow_summary_data_load.py': 'diagnostics_logs_json_data.json',
    'import_physical_query_response_data_load.py': 'diagnostics_logs_json_data.json',
    'import_physical_query_details_data_load.py': 'diagnostics_logs_json_data.json',
    'import_logical_query_summary_data_load.py': 'diagnostics_logs_json_data.json',
    'import_init_logs_data_load.py': 'init_blocks_logs_json_data.json',
    'import_physical_query_log_data_load.py': 'diagnostics_logs_json_data.json'
}

# Read data and failed data directories from config.ini
data_dir = config['DATA']['data_dir']
failed_json_dir = config['FAILED']['data_dir']

def execute_python_script(script):
    script_name = os.path.basename(script)
    logging.info(f"Running the {script_name}...")
    try:
        result = subprocess.run(["python3", script], stdout=subprocess.PIPE, stderr=subprocess.PIPE, check=True)
        logging.info(f"{script_name} completed successfully.")
        return True
    except subprocess.CalledProcessError as e:
        logging.error(f"{script_name} failed: {e.stderr.decode().strip()}")
        return False
    except Exception as e:
        logging.error(f"{script_name} failed: {str(e)}")
        return False

def run_export_scripts():
    load_type = config['LOAD']['load_type'].lower()
    if load_type == 'full_load':
        for script in export_scripts:
            result = execute_python_script(script)
            if not result:
                return False
        return True
    elif load_type == 'incremental':
        with concurrent.futures.ThreadPoolExecutor() as executor:
            download_results = list(executor.map(execute_python_script, export_scripts))
        return all(download_results)
    else:
        logging.error("Invalid load_type specified in config.ini. Please specify either 'full_load' or 'incremental'")
        return False

def run_import_scripts():
    import_results = {}
    for script, json_file in import_scripts.items():
        result = execute_python_script(script)
        import_results[script] = result
    return import_results

def main():
    LD_LIBRARY_PATH = config['PATHS']['instantclient_dir']
    os.environ['LD_LIBRARY_PATH'] = LD_LIBRARY_PATH

    export_success = run_export_scripts()
    import_results = run_import_scripts()
    
    # Checking the failed scripts and corresponding json data files
    failed_json_files = []
    for script, success in import_results.items():
        if not success:
            json_file = import_scripts.get(script)
            if json_file:
                failed_json_files.append(json_file)
    
    # Move failed JSON files to the failed directory and delete json data files which are successfully loaded into the database. Please fix the errors and try load the failed JSON files again manually.
    if failed_json_files:
        timestamp = datetime.now().strftime("%Y%m%d%H%M%S")
        for json_file in failed_json_files:
            json_path = os.path.join(data_dir, json_file)
            if os.path.exists(json_path):
                new_file_name = f"{os.path.splitext(os.path.basename(json_file))[0]}_{timestamp}.json"
                failed_json_path = os.path.join(failed_json_dir, new_file_name)
                shutil.move(json_path, failed_json_path)
                logging.error(f"Moved {json_path} to {failed_json_path} due to import failure.")
    
    json_files = glob.glob(os.path.join(data_dir, '*.json'))
    for json_file in json_files:
        os.remove(json_file)
        logging.info(f"Deleted {json_file}.")
    
    if failed_json_files:
        logging.error("Data load has failed. Please check the logs.")
        logging.info("**************************************Data load refresh has failed*******************************************")
        print("Data load has failed. Please check the logs.")
    else:
        logging.info("All import scripts executed successfully.")
        logging.info("********************************Data loads refresh has been successfully finished****************************")
        print("Data load completed successfully.")
if __name__ == "__main__":
    main()