#!/bin/bash

#-----------------------------------------------------------------------
# Enter Parameter Values.
#-----------------------------------------------------------------------
User=ohs
Group=ohs
Software_Location=/tmp/softwares
Install_Base_Location=/scratch/ohs
JDK_Version="1.8.0_411"
Wallet_Password=Oracle@123
# Enter Load Balancer or OHS (Whichever is the FrontEnd) DNS Name e.g oasdev.oracleceal.com
# e.g. oasdev.oracleceal.com:PortNo, enter PortNo if its other than 443. e.g. oasedev.oracleceal.com:4443
LoadBalancerDNS=oasdev.oracleceal.com
# Enter the Hostname or IP Address of the Backend OAS Server and Port.
OAS_Hostname=10.0.0.203
OAS_PortNo=9502
# Below 2 Parameters are Optional.
SAML_IDP_Logout_URL=
Post_Logout_URL=
# Are you OffLoading SSL at the OHS Server? true|false.
# SSL OffLoading at the OHS Server means that the Backend OAS Server is on Non-SSL Ports.
SSL_OffLoading=true
OHS_HTTP_Port=7777
OHS_HTTPS_Port=4443
#-----------------------------------------------------------------------
# Place the below file in the path $Software_Location/ohsFiles/certs
#-----------------------------------------------------------------------
LB_Certificate=oasdev.crt
LB_PrivateKey=oasdev.key
# Skip provinding the password if your PrivateKey doesn't have a password. 
LB_PrivateKey_Password=
LB_CA_Intermediate=CAInter.crt
LB_CA_Root=CARoot.crt
# Specify the PFX File name if you have it. Skip the next 2 lines if you dont have.
PFXFile=
PFX_Password=
# Specify the JKS keystore File name if you have it. Skip the next 2 lines if you dont have.
JKSFile=
JKS_Password=
#-----------------------------------------------------------------------


prequisiteCheck()
{
 # Check if the user is root
 if [ "$(whoami)" != "$User" ]; then
    echo "This script must be run as $User"
    exit 1
 fi
 
 echo "Prerequisites Checking....."
 
 # Check if the required Load Balancer certificates and privatekey uploaded to the ohsFiles folder.
 if [ -f $Software_Location/ohsFiles/certs/$LB_Certificate ] && [ -f $Software_Location/ohsFiles/certs/$LB_PrivateKey ] && [ -f $Software_Location/ohsFiles/certs/$LB_CA_Intermediate ] && [ -f $Software_Location/ohsFiles/certs/$LB_CA_Root ]; then
    echo "Prerequisites to create a wallet using the Load Balancer certificates are satisfied."
 else
    echo "Few of the files required to create OHS wallet are missing, Exiting....."
	exit 1
 fi 
 
 echo "Some files might have unexpected characters like carriage returns etc when files are moved from Windows to Linux."
 echo "You need to remove the unwanted characters using dos2unix command." 
	
 # Check if dos2unix package is installed or not.
 if which dos2unix >/dev/null; then
    dos2unix $Software_Location/*
 else
	echo "dos2unix is not installed, Exiting....."
	exit 1
 fi

}


# Create OHS Wallet, Refer: https://blogs.oracle.com/analytics/post/oas-mp-end-to-end-ssl.

# Create a PFX File using the Load Balancer Certificate and Private Key.
# If you already have the PFX file, will skip this step.
createPFX()
{
 echo "Creating a PFX file using the Load Balancer Certificate and the Private Key."
 cd $Software_Location/ohsFiles/certs
 PFXFile=${LoadBalancerDNS%%.*}.pfx
 
 if [ -n "$LB_PrivateKey_Password" ]; then
	# If the Private Key has a Password, Remove it using the below command.
	openssl rsa -in $LB_PrivateKey -out TempPrivateKey.key -passin pass:$LB_PrivateKey_Password
	rm -f $LB_PrivateKey
	mv TempPrivateKey.key $LB_PrivateKey
 fi
  
 PFX_Password=$Wallet_Password
 openssl pkcs12 -export -out $PFXFile -inkey $LB_PrivateKey -in $LB_Certificate -passout pass:$PFX_Password
}


# Creating the JKS keystore. If you already have a JKS keystore, will skip this step.
createJKS()
{
 if [ -z "$PFXFile" ]; then
	echo "PFX File is not provided, creating it using the provided LB Certificates."
	createPFX
 fi
 
 echo "Creating a JKS keystore using the PFX File, CA Intermediate and CA Root Certificates."
 cd $Software_Location/ohsFiles/certs
 
 export JAVA_HOME=$Install_Base_Location/jdk$JDK_Version
 export PATH=$JAVA_HOME/bin:$PATH
 JKSFile=${LoadBalancerDNS%%.*}.jks
 JKS_Password=$Wallet_Password
 
 # Creating JKS keystore from the PFX file.
 # If you already have the JKS keystore, skip this command. Comment the next line.
 $JAVA_HOME/bin/keytool -importkeystore -srckeystore $PFXFile -srcstoretype pkcs12 -srcstorepass $PFX_Password -destkeystore $JKSFile -deststoretype JKS -deststorepass $JKS_Password -noprompt
 
 # Importing CAInter and CARoot Certificates into the JKS keystore.
 $JAVA_HOME/bin/keytool -importcert -alias CAInter -file $LB_CA_Intermediate -keystore $JKSFile -storepass $JKS_Password -noprompt
 $JAVA_HOME/bin/keytool -importcert -alias CARoot -file $LB_CA_Root -keystore $JKSFile -storepass $JKS_Password -noprompt
 
 # Creating a Chain Certificate by appending Load Balancer + CAInter + CARoot certificates in sequence.
 cat $LB_Certificate > ${LoadBalancerDNS%%.*}_chain.crt
 cat $LB_CA_Intermediate >> ${LoadBalancerDNS%%.*}_chain.crt
 cat $LB_CA_Root >> ${LoadBalancerDNS%%.*}_chain.crt
 
 # Sometimes the appending can have carriage return, unwanted characters at the end of each line, need to convert to unix format.
 dos2unix ${LoadBalancerDNS%%.*}_chain.crt
 
 # Sometimes the appending of files can be wrongly placed like e.g. -----END CERTIFICATE----------BEGIN CERTIFICATE-----
 # Check for such errors and correct them.
 sed -i "s|-----END CERTIFICATE----------BEGIN CERTIFICATE-----|-----END CERTIFICATE-----\n-----BEGIN CERTIFICATE-----|g" ${LoadBalancerDNS%%.*}_chain.crt
 
 # Importing the Load Balancer Server Chain Certificate in to the JKS keystore to establish the chain in the keystore.
 $JAVA_HOME/bin/keytool -importcert -alias "1" -file ${LoadBalancerDNS%%.*}_chain.crt -keystore $JKSFile -storepass $JKS_Password -noprompt
 
 # List the certificates in the JKS keystore.
 $JAVA_HOME/bin/keytool -list -v -keystore $JKSFile -storepass $JKS_Password > JKSList.txt
 
 if grep -q "Alias name: caroot" JKSList.txt; then
    echo "Successfully Imported CARoot Certificate into the JKS keystore."
 else
	echo "Failed to list caroot certificate in the JKS keystore, check if it is imported."
	exit 1
 fi
 
 if grep -q "Alias name: cainter" JKSList.txt; then
    echo "Successfully Imported CAInter Certificate into the JKS keystore."
 else
	echo "Failed to list cainter certificate in the JKS keystore, check if it is imported."
	exit 1
 fi
 
 if grep -q "Certificate chain length: 3" JKSList.txt; then
    echo "Successfully Imported Chain Load Balancer or Server Certificate into the JKS keystore."
 else
	echo "Failed to import the chain server certificate in the JKS keystore, check if it is imported properly."
	exit 1
 fi

 echo "Successfully created the JKS keystore using the provided certificates." 
}


# Convert the JKS keystore to a OHS Wallet.
createWallet()
{
 if [ -z "$JKSFile" ]; then
	echo "JKS keystore File is not provided, creating it using the PFX file."
	createJKS
 fi
 
 echo "Creating a Wallet for OHS using the JKS keystore."

 cd $Software_Location/ohsFiles/certs
 mkdir eWallet
 $Install_Base_Location/Oracle/Middleware/Oracle_Home/oracle_common/bin/orapki wallet create -wallet ./eWallet/ -auto_login -pwd $Wallet_Password
 
 $Install_Base_Location/Oracle/Middleware/Oracle_Home/oracle_common/bin/orapki wallet jks_to_pkcs12 -wallet ./eWallet/ -pwd $Wallet_Password -keystore $JKSFile -jkspwd $JKS_Password
 
 $Install_Base_Location/Oracle/Middleware/Oracle_Home/oracle_common/bin/orapki wallet display -wallet ./eWallet/ -pwd $Wallet_Password

 echo "Successfully created the OHS Wallet, using the provided JKS keystore." 
}


# Configure OHS file for SAML SSO enablement.
configureOHS()
{
 echo "Configuring OHS for SAML SSO enablement."
 
 OHS_DOMAIN_HOME=$Install_Base_Location/Oracle/Middleware/Oracle_Home/user_projects/domains/ohs_domain
 OHS_CONFIG_HOME=$OHS_DOMAIN_HOME/config/fmwconfig/components/OHS/ohs1
 
 cp $Software_Location/ohsFiles/compressNcache.conf $OHS_CONFIG_HOME 
 cp $Software_Location/ohsFiles/analytics.conf $OHS_CONFIG_HOME
 cp $Software_Location/ohsFiles/analyticsclustered.conf $OHS_CONFIG_HOME 
 cp $Software_Location/ohsFiles/workers.conf $OHS_CONFIG_HOME
 cp $Software_Location/ohsFiles/redirect_http_to_https.conf $OHS_CONFIG_HOME
 cp $Software_Location/ohsFiles/psr.conf $OHS_CONFIG_HOME
 cp $Software_Location/ohsFiles/logout.html $OHS_CONFIG_HOME/htdocs
 mkdir -p $OHS_DOMAIN_HOME/../../../SSL
 cp -R $Software_Location/ohsFiles/certs/eWallet $OHS_DOMAIN_HOME/../../../SSL 
 OHS_WALLET_PATH=$(cd "$OHS_DOMAIN_HOME/../../../SSL/eWallet" && pwd) 
 
 # Set logout.html page
 sed -i "s|<apache-or-loadbalancer-hostname.com>|$LoadBalancerDNS|g" $OHS_CONFIG_HOME/htdocs/logout.html
 if [ -n $SAML_IDP_Logout_URL ]; then
	echo "SAML IDP Logout URL defined, Configuring it."
	sed -i "s|<IdP-logout-url>|$SAML_IDP_Logout_URL|g" $OHS_CONFIG_HOME/htdocs/logout.html
 fi
 
 if [ -n $Post_Logout_URL ]; then
	echo "Post Logout URL defined, Configuring it."
	sed -i "s|<Post-Logout-URL>|$Post_Logout_URL|g" $OHS_CONFIG_HOME/htdocs/logout.html
 else
	echo "Post Logout URL not defined, Configuring default Post Logout URL."
	PLURL="https:\/\/${LoadBalancerDNS}\/dv"
	sed -i "s|<Post-Logout-URL>|$PLURL|g" $OHS_CONFIG_HOME/htdocs/logout.html
 fi
 
 # Set the Backend OAS Server Hostname and Port in the analytics.conf file. 
 if [ -n $OAS_Hostname ] && [ -n $OAS_PortNo ]; then
	echo "Configuring analytics.conf file to set OAS Hostname and Port."
	sed -i "s|<oas-server-hostname>|$OAS_Hostname|g" $OHS_CONFIG_HOME/analytics.conf
	sed -i "s|<oas-server-port>|$OAS_PortNo|g" $OHS_CONFIG_HOME/analytics.conf
 else
	echo "OAS Hostname and Port Parameters are not set, Existing....."
	exit 1
 fi
 
 # Load required Files in httpd.conf file.
 cp $OHS_CONFIG_HOME/httpd.conf $OHS_CONFIG_HOME/httpd.conf.original
 sed -i "s|Timeout 60|ProxyTimeout 6000\nTimeout 3600|g" $OHS_CONFIG_HOME/httpd.conf
 sed -i "s|KeepAliveTimeout 5|KeepAliveTimeout 61|g" $OHS_CONFIG_HOME/httpd.conf
 sed -i "s|MaxKeepAliveRequests 100|MaxKeepAliveRequests 500|g" $OHS_CONFIG_HOME/httpd.conf
 echo "# Adding SSO Configuration." >> $OHS_CONFIG_HOME/httpd.conf
 echo 'LoadModule expires_module "${PRODUCT_HOME}/modules/mod_expires.so"' >> $OHS_CONFIG_HOME/httpd.conf
 echo 'LoadModule deflate_module "${PRODUCT_HOME}/modules/mod_deflate.so"' >> $OHS_CONFIG_HOME/httpd.conf
 echo 'include "analytics.conf"' >> $OHS_CONFIG_HOME/httpd.conf
 echo 'include "compressNcache.conf"' >> $OHS_CONFIG_HOME/httpd.conf
 echo 'include "psr.conf"' >> $OHS_CONFIG_HOME/httpd.conf
 # Use either the redirect_http_to_https.conf file or the Header edit Location.
 echo 'include "redirect_http_to_https.conf"' >> $OHS_CONFIG_HOME/httpd.conf
 #echo 'Header edit Location ^http://(.*)$ https://$1' >> $OHS_CONFIG_HOME/httpd.conf
 
 # Configure ssl.conf 
 cp $OHS_CONFIG_HOME/ssl.conf $OHS_CONFIG_HOME/ssl.conf.original
 sed -i "s|SSLWallet|#SSLWallet|g" $OHS_CONFIG_HOME/ssl.conf
 add_lines1="ServerName ${LoadBalancerDNS}\n   SSLEngine on\n   SSLProxyEngine on\n   RequestHeader set WL-Proxy-SSL \"true\"\n   RequestHeader set IS_SSL \"ssl\"\n   RewriteEngine On\n   RewriteOptions Inherit\n\n   #UnComment_Below_Lines_If_SSL_enabled_at_OAS\n   #SSLVerifyClient none\n   #SSLProxyVerify none\n   #SSLProxyCheckPeerName off\n   #SSLProxyCheckPeerCN off\n   #SSLProxyCheckPeerExpire off\n"
 sed -i "s|SSLEngine on|$add_lines1|g" $OHS_CONFIG_HOME/ssl.conf
 add_lines2="#Path to the wallet\n   SSLWallet \"$OHS_WALLET_PATH\"\n   SSLProxyWallet \"$OHS_WALLET_PATH\""
 sed -i "s|#Path to the wallet|$add_lines2|g" $OHS_CONFIG_HOME/ssl.conf
 
 echo "Successfully configured OHS for Single Sign-On." 
}


restartOHS()
{
 echo "Restarting OHS after the SSO Configuration."
 $OHS_DOMAIN_HOME/bin/restartComponent.sh ohs1
}


testOHS()
{
 OHS_Host=`hostname -f`
 curl -Is "http://${OHS_Host}:${OHS_HTTP_Port}/" | head -n 1 > $Software_Location/result.txt

 if grep -q "HTTP/1.1 301 Moved Permanently" $Software_Location/result.txt; then
    echo "OHS is restarted Successfully!!!."
	cleanUP
 else
    echo "Unable to restart OHS, Check manually!!!"
	echo "Exiting....."
	exit 1
 fi
}


cleanUP()
{
 rm -f $Software_Location/ohsFiles/certs/JKSList.txt
 rm -f $Software_Location/result.txt
}


SPMetadata()
{
 echo ""
 echo "====================================================================================================================="
 echo ""
 echo " The Oracle HTTP Server with Shibboleth-SP is Configured."
 echo " Use below Service Provider End Points to Configure with the SAML Identity Provider."
 echo ""
 echo " EntityID: https://${LoadBalancerDNS}/analytics/shibboleth"
 echo ""
 echo " AssertionConsumerService: https://${LoadBalancerDNS}/Shibboleth.sso/SAML2/POST"
 echo ""
 echo " SingleLogoutService: https://${LoadBalancerDNS}/Shibboleth.sso/SLO/POST"
 echo ""
 echo " Signing or Encryption Certificate:"
 echo ""
 echo " Use the same Load Balancer Certificate placed in the folder: $Software_Location/ohsFiles/certs"
 echo ""
 echo " Displaying the SP Signing Certificate for conveniency."
 echo ""
 cat $Software_Location/ohsFiles/certs/$LB_Certificate
 echo ""
 echo ""
 echo "*********************************************************************************************************************"
 echo ""
 echo " To download the SAML Service Provider Metadata Use the below URL after configuring the Load Balancer."
 echo ""
 echo " Download the Service Provider Metadata: https://${LoadBalancerDNS}/Shibboleth.sso/Metadata"
 echo ""
 echo "*********************************************************************************************************************" 
 echo ""
 echo "====================================================================================================================="
 echo ""
}


# Call all Functiona
prequisiteCheck
createWallet
configureOHS
restartOHS
testOHS
cleanUP
SPMetadata