#!/bin/bash

#-------------------------------------------------------------------------
# Enter Variable Values
#-------------------------------------------------------------------------
# Specify the user:group installing OAS
User=ohs
Group=ohs
Software_Location=/tmp/softwares
Install_Base_Location=/scratch/ohs
# Example value of JDK Version : "1.8.0_411"
JDK_Version="1.8.0_411"
JDK_FileName=jdk-8u411-linux-x64.tar.gz
OHS_FileName=fmw_12.2.1.4.0_ohs_linux64_Disk1_1of1.zip
# Specify the Operating System release like 7 or 8 for OEL/RHEL.
OS_Release=8
# If the OS Version is 8
# Follow Support Doc ID: 2652061.1 and apply the patches 31190532 and 31090393.
OL8_Patch1=p31190532_122140_Generic.zip
OL8_Patch2=p31090393_122140_Linux-x86-64.zip
#-------------------------------------------------------------------------


# Check if all the required software files are available.
prerequisiteCheck()
{
 # Check if the user is oracle
 if [ "$(whoami)" != "$User" ]; then
	echo "This script must be run as $User user"
    exit 1
 fi

 # Check if the OHS Installation required Softwares are available in the required Location or not.
 if [[ -f $Software_Location/$JDK_FileName ]]; then
    echo "JDK - Prerequisite Satisfied."
 else
    echo "JDK - Required file does not exist, Exiting....."
	exit 1
 fi

 if [[ -f $Software_Location/$OHS_FileName ]]; then
    echo "OHS - Prerequisite Satisfied."
 else
    echo "OHS - Required file does not exist, Exiting....."
	exit 1
 fi

 if [[ $OS_Release == 8 ]]; then
	 if [[ -f $Software_Location/$OL8_Patch1 ]]; then
    		echo "OHS - Prerequisite Satisfied for OS version 8."
 	 else
    		echo "OHS - OS version 8 Patch1 file does not exist, Exiting....."
        	exit 1
 	 fi
	
	 if [[ -f $Software_Location/$OL8_Patch2 ]]; then
    		echo "OHS - Prerequisite Satisfied for OS version 8."
 	 else
    		echo "OHS - OS version 8 Patch2 file does not exist, Exiting....."
        	exit 1
 	 fi
 else 
	 echo "OS version is 7, ignoring the patch file check."
 fi

 echo "--------------------------------------------------------------------------------------------------"
 echo "Software required to install Oracle HTTP Server are available, proceeding to install....."
 echo "OHS Silent Installation....."
 echo "--------------------------------------------------------------------------------------------------"
}


# Install JDK8 and OraInventory.
installJDK()
{
 echo "Installing JDK8 and Creating oraInventory....."
 
 # Create OHS Install Location.
 sudo mkdir -p $Install_Base_Location
 sudo chown -R $User:$Group $Install_Base_Location

 # Install JDK
 cd $Install_Base_Location
 tar -xzf $Software_Location/$JDK_FileName 

 # Inventory File
 mkdir -p $Install_Base_Location/oraInventory
 sudo chown -R $User:$Group $Install_Base_Location/oraInventory
 echo "inventory_loc=$Install_Base_Location/oraInventory
 inst_group=$Group" > $Install_Base_Location/oraInventory/oraInst.loc

 # Environment Variables
 export JAVA_HOME=$Install_Base_Location/jdk$JDK_Version
 export PATH=$JAVA_HOME/bin:$PATH
 
 # Check Installation
 java_version=$(java -version 2>&1 | awk '/version/ {print $3}' | sed 's/"//g')
 if [[ $java_version == $JDK_Version ]]; then
	echo "JDK $JDK_Version installed Successfully."
 else
	echo "JDK installation failed."
	exit 1
 fi
}


# Install FMW required binaries for OEL7, OEL8 etc.
# https://docs.oracle.com/en/middleware/fusion-middleware/12.2.1.4/sysrs/system-requirements-and-specifications.html#GUID-37C51062-3732-4A4B-8E0E-003D9DFC8C26

installLibraries()
{
 if [[ $OS_Release == 7 ]]; then
	echo "Installing the FMW 12.2.1.4 required Libraries for Linux 7 Release....."
	# FMW required Binaries for OEL7/RHEL7.
	sudo yum install -y binutils*
	sudo yum install -y compat-libcap*
	sudo yum install -y compat-libstdc++*
	sudo yum install -y gcc*
	sudo yum install -y gcc-c++*
	sudo yum install -y glibc*
	sudo yum install -y glibc-devel*
	sudo yum install -y libaio*
	sudo yum install -y libaio-devel*
	sudo yum install -y libgcc*
	sudo yum install -y libstdc++*
	sudo yum install -y libstdc++-devel*
	sudo yum install -y dejavu-serif-fonts
	sudo yum install -y ksh
	sudo yum install -y make*
	sudo yum install -y sysstat* 
	sudo yum install -y numactl*
	sudo yum install -y numactl-devel*
	sudo yum install -y motif*
	sudo yum install -y motif-devel*
	sudo yum install -y redhat-lsb*
	sudo yum install -y redhat-lsb-core*
	sudo yum install -y openssl*
	sudo yum install -y xorg-x11-utils*
 elif [[ $OS_Release == 8 ]]; then
	echo "Installing the FMW 12.2.1.4 required Libraries for Linux 8 Release....."
	# FMW required Binaries for OEL8/RHEL8.
	sudo dnf install -y binutils*
	sudo dnf install -y gcc*
	sudo dnf install -y gcc-c++*
	sudo dnf install -y glibc*
	sudo dnf install -y glibc-devel*
	sudo dnf install -y libaio*
	sudo dnf install -y libaio-devel*
	sudo dnf install -y libgcc*
	sudo dnf install -y libstdc++*
	sudo dnf install -y libstdc++-devel*
	sudo dnf install -y libnsl*
	sudo yum install -y ksh
	sudo dnf install -y sysstat*
	sudo dnf install -y motif*
	sudo dnf install -y motif-devel*
	sudo dnf install -y redhat-lsb*
	sudo dnf install -y redhat-lsb-core*
	sudo dnf install -y openssl*
	sudo dnf install -y make*
	sudo dnf install -y xorg-x11-utils*
 else
	echo "Unable to determine the OS Linux release....."
	exit 1
 fi
}


# Create the OHS Install Response File.
createOHSRespone()
{
 echo "Creating Silent Install Response File for FMW."
 echo "[ENGINE]
Response File Version=1.0.0.0.0
[GENERIC]
ORACLE_HOME=$Install_Base_Location/Oracle/Middleware/Oracle_Home
INSTALL_TYPE=Standalone HTTP Server (Managed independently of WebLogic server)
DECLINE_SECURITY_UPDATES=true
SECURITY_UPDATES_VIA_MYORACLESUPPORT=false" > $Software_Location/ohs_12214_rsp_file.rsp
}


# Oracle HTTP Server Silent Installation
installOHS()
{
 if [[ $OS_Release == 7 ]]; then
 	echo "Installing OHS in Silent Mode....."
 	mkdir -p $Software_Location/ohs
 	unzip $Software_Location/$OHS_FileName -d $Software_Location/ohs
 	sudo chown -R $User:$Group $Software_Location/ohs
 	$Software_Location/ohs/fmw_12.2.1.4.0_ohs_linux64.bin -jreLoc $JAVA_HOME -silent -responseFile $Software_Location/ohs_12214_rsp_file.rsp -invPtrLoc $Install_Base_Location/oraInventory/oraInst.loc
 	rm -rf $Software_Location/ohs
 elif [[ $OS_Release == 8 ]]; then
	echo "Installing OHS in Silent Mode....."
	mkdir -p $Software_Location/ohs
	unzip $Software_Location/$OHS_FileName -d $Software_Location/ohs
	sudo chown -R $User:$Group $Software_Location/ohs
	cd $Software_Location
	unzip $Software_Location/$OL8_Patch1 
	P1_Folder=`echo "$OL8_Patch1" | sed 's/^.*p\([0-9]*\)_.*/\1/'`
	$Software_Location/ohs/fmw_12.2.1.4.0_ohs_linux64.bin -prereqConfigLoc $Software_Location/$P1_Folder/prereq_metadata/oracle.as.install.ohs.prerequisite/prereq -jreLoc $JAVA_HOME -silent -responseFile $Software_Location/ohs_12214_rsp_file.rsp -invPtrLoc $Install_Base_Location/oraInventory/oraInst.loc
	rm -rf $Software_Location/ohs
	echo "Applying the patch 31090393....."
	cd $Software_Location
	unzip $Software_Location/$OL8_Patch2
	P2_Folder=`echo "$OL8_Patch2" | sed 's/^.*p\([0-9]*\)_.*/\1/'`
	cd $P2_Folder
	yes | yes | $Install_Base_Location/Oracle/Middleware/Oracle_Home/OPatch/opatch apply
 else
 	echo "The OS version is neither 7 or 8 please check the OS version and run the script again."
 fi
 
 # Check if the Oracle_Home Folder Exists
 if [[ -d "$Install_Base_Location/Oracle/Middleware/Oracle_Home" ]]; then
	echo "Oracle_Home exists, OHS installed successfully."
 else
	echo "Unable to install OHS."
	exit 1
 fi
}


# Clean the Files
cleanUP()
{
 echo "Cleaning up the files used in the Silent Installation....."
 rm -f $Software_Location/$JDK_FileName
 rm -f $Software_Location/$OHS_FileName
 rm -f $Software_Location/ohs_12214_rsp_file.rsp
 rm -f $Software_Location/$OL8_Patch1
 rm -f $Software_Location/$OL8_Patch2
 rm -rf $Software_Location/$P1_Folder
 rm -rf $Software_Location/$P2_Folder
}


# Call the Functions
prerequisiteCheck
installJDK
installLibraries
createOHSRespone
installOHS
cleanUP
