#!/bin/bash

#------------------------------------------------------------------------------------------------
# Enter Variable Values
#------------------------------------------------------------------------------------------------
# Specify the user:group installing OAS
User=oracle
Group=oinstall
Software_Location=/tmp/softwares
Install_Base_Location=/u01
# Example value of JDK Version : "1.8.0_411"
JDK_Version="1.8.0_411"
JDK_FileName=jdk-8u411-linux-x64.tar.gz
FMW_FileName=fmw_12.2.1.4.0_infrastructure_Disk1_1of1.zip
OAS_FileName=V1041086-01.zip
# Specify the Operating System release like 7 or 8 for OEL/RHEL.
OS_Release=8
#------------------------------------------------------------------------------------------------


# Check if all the required software files are available.
prerequisiteCheck()
{
 # Check if the user is oracle
 if [ "$(whoami)" != "$User" ]; then
	echo "This script must be run as $User user"
    exit 1
 fi

 # Check if the OAS Installation required Softwares and Patches are available in the required Location or not.
 if [[ -f $Software_Location/$JDK_FileName ]]; then
    echo "JDK - Prerequisite Satisfied."
 else
    echo "JDK - Required file does not exist, Exiting....."
	exit 1
 fi

 if [[ -f $Software_Location/$FMW_FileName ]]; then
    echo "FMW - Prerequisite Satisfied."
 else
    echo "FMW - Required file does not exist, Exiting....."
	exit 1
 fi

 if [[ -f $Software_Location/$OAS_FileName ]]; then
    echo "OAS - Prerequisite Satisfied."
 else
    echo "OAS - Required file does not exist, Exiting....."
	exit 1
 fi

 echo "----------------------------------------------------------------------------------------------------------------"
 echo "Software required to install Oracle Analytics Server 2024 are available, proceeding to install....."
 echo "Below steps will be performed for a Single Node OAS Server:"
 echo "OAS Silent Installation....."
 echo "----------------------------------------------------------------------------------------------------------------"
}


# Install JDK8 and OraInventory.
installJDK()
{
 echo "Installing JDK8 and Creating oraInventory....."
 
 # Create OAS Install Location.
 sudo mkdir -p $Install_Base_Location/app
 sudo chown -R $User:$Group $Install_Base_Location

 # Install JDK
 cd $Install_Base_Location/app
 tar -xzf $Software_Location/$JDK_FileName 

 # Inventory File
 mkdir -p $Install_Base_Location/app/oraInventory
 sudo chown -R $User:$Group $Install_Base_Location/app/oraInventory
 echo "inventory_loc=$Install_Base_Location/app/oraInventory
 inst_group=$Group" > $Install_Base_Location/app/oraInventory/oraInst.loc

 # Environment Variables
 export JAVA_HOME=$Install_Base_Location/app/jdk$JDK_Version
 export PATH=$JAVA_HOME/bin:$PATH
 
 # Check Installation
 java_version=$(java -version 2>&1 | awk '/version/ {print $3}' | sed 's/"//g')
 if [[ $java_version == $JDK_Version ]]; then
	echo "JDK $JDK_Version installed Successfully."
 else
	echo "JDK installation failed."
	exit 1
 fi
}


# Install FMW and OAS required binaries for OEL7, OEL8 etc.
# https://docs.oracle.com/en/middleware/fusion-middleware/12.2.1.4/sysrs/system-requirements-and-specifications.html#GUID-37C51062-3732-4A4B-8E0E-003D9DFC8C26

installLibraries()
{
 if [[ $OS_Release == 7 ]]; then
	echo "Installing the FMW 12.2.1.4 required Libraries for Linux 7 Release....."
	# OAS ML Binary
	sudo yum install -y libgfortran*
	# OAS and FMW required Binaries for OEL7/RHEL7.
	sudo yum install -y binutils*
	sudo yum install -y compat-libcap*
	sudo yum install -y compat-libstdc++*
	sudo yum install -y gcc*
	sudo yum install -y gcc-c++*
	sudo yum install -y glibc*
	sudo yum install -y glibc-devel*
	sudo yum install -y libaio*
	sudo yum install -y libaio-devel*
	sudo yum install -y libgcc*
	sudo yum install -y libstdc++*
	sudo yum install -y libstdc++-devel*
	sudo yum install -y dejavu-serif-fonts
	sudo yum install -y ksh
	sudo yum install -y make*
	sudo yum install -y sysstat* 
	sudo yum install -y numactl*
	sudo yum install -y numactl-devel*
	sudo yum install -y motif*
	sudo yum install -y motif-devel*
	sudo yum install -y redhat-lsb*
	sudo yum install -y redhat-lsb-core*
	sudo yum install -y openssl*
	sudo yum install -y xorg-x11-utils*
 elif [[ $OS_Release == 8 ]]; then
	echo "Installing the FMW 12.2.1.4 required Libraries for Linux 8 Release....."
	# OAS ML Binary
	sudo dnf install -y compat-libgfortran*
	# OAS and FMW required Binaries for OEL8/RHEL8.
	sudo dnf install -y binutils*
	sudo dnf install -y gcc*
	sudo dnf install -y gcc-c++*
	sudo dnf install -y glibc*
	sudo dnf install -y glibc-devel*
	sudo dnf install -y libaio*
	sudo dnf install -y libaio-devel*
	sudo dnf install -y libgcc*
	sudo dnf install -y libstdc++*
	sudo dnf install -y libstdc++-devel*
	sudo dnf install -y libnsl*
	sudo dnf install -y sysstat*
	sudo dnf install -y motif*
	sudo dnf install -y motif-devel*
	sudo dnf install -y redhat-lsb*
	sudo dnf install -y redhat-lsb-core*
	sudo dnf install -y openssl*
	sudo dnf install -y make*
	sudo dnf install -y xorg-x11-utils*
 else
	echo "Unable to determine the OS Linux release....."
	exit 1
 fi
}


# Create the FMW Install Response File.
createFMWRespone()
{
 echo "Creating Silent Install Response File for FMW."
 
 echo "[ENGINE]

#DO NOT CHANGE THIS.
Response File Version=1.0.0.0.0

[GENERIC]

#Set this to true if you wish to skip software updates
DECLINE_AUTO_UPDATES=true

#My Oracle Support User Name
MOS_USERNAME=

#My Oracle Support Password
MOS_PASSWORD=

#If the Software updates are already downloaded and available on your local system, then specify the path to the directory where these patches are available and set SPECIFY_DOWNLOAD_LOCATION to true
AUTO_UPDATES_LOCATION=

#Proxy Server Name to connect to My Oracle Support
SOFTWARE_UPDATES_PROXY_SERVER=

#Proxy Server Port
SOFTWARE_UPDATES_PROXY_PORT=

#Proxy Server Username
SOFTWARE_UPDATES_PROXY_USER=

#Proxy Server Password
SOFTWARE_UPDATES_PROXY_PASSWORD=

#The oracle home location. This can be an existing Oracle Home or a new Oracle Home
ORACLE_HOME=$Install_Base_Location/app/Oracle/Middleware/Oracle_Home

#Set this variable value to the Installation Type selected. e.g. Fusion Middleware Infrastructure, Fusion Middleware Infrastructure With Examples.
INSTALL_TYPE=Fusion Middleware Infrastructure

#Provide the My Oracle Support Username. If you wish to ignore Oracle Configuration Manager configuration provide empty string for user name.
MYORACLESUPPORT_USERNAME=

#Provide the My Oracle Support Password
MYORACLESUPPORT_PASSWORD=

#Set this to true if you wish to decline the security updates. Setting this to true and providing empty string for My Oracle Support username will ignore the Oracle Configuration Manager configuration
DECLINE_SECURITY_UPDATES=true

#Set this to true if My Oracle Support Password is specified
SECURITY_UPDATES_VIA_MYORACLESUPPORT=false

#Provide the Proxy Host
PROXY_HOST=

#Provide the Proxy Port
PROXY_PORT=

#Provide the Proxy Username
PROXY_USER=

#Provide the Proxy Password
PROXY_PWD=

#Type String (URL format) Indicates the OCM Repeater URL which should be of the format [scheme[Http/Https]]://[repeater host]:[repeater port]
COLLECTOR_SUPPORTHUB_URL=" > $Software_Location/fmw_12214_rsp_file.rsp
}


# Fusion Middleware Infrastructure Installation
installFMW()
{
 echo "Installing FMW in Silent Mode....."
 mkdir -p $Software_Location/fmw
 unzip $Software_Location/$FMW_FileName -d $Software_Location/fmw
 sudo chown -R $User:$Group $Software_Location/fmw
 java -jar $Software_Location/fmw/fmw_12.2.1.4.0_infrastructure.jar -silent -responseFile $Software_Location/fmw_12214_rsp_file.rsp -invPtrLoc $Install_Base_Location/app/oraInventory/oraInst.loc
 rm -rf $Software_Location/fmw

 # Check if the Oracle_Home Folder Exists
 if [[ -d "$Install_Base_Location/app/Oracle/Middleware/Oracle_Home" ]]; then
	echo "Oracle_Home exists, FMW installed successfully."
 else
	echo "Unable to install FMW."
	exit 1
 fi
}


# Create OAS Install Response File.
createOASResponse()
{
 echo "Creating Silent Install Response File for OAS."
 echo "[ENGINE]

#DO NOT CHANGE THIS.
Response File Version=1.0.0.0.0

[GENERIC]

#Set this to true if you wish to skip software updates
DECLINE_AUTO_UPDATES=true

#My Oracle Support User Name
MOS_USERNAME=

#My Oracle Support Password
MOS_PASSWORD=

#If the Software updates are already downloaded and available on your local system, then specify the path to the directory where these patches are available and set SPECIFY_DOWNLOAD_LOCATION to true
AUTO_UPDATES_LOCATION=

#Proxy Server Name to connect to My Oracle Support
SOFTWARE_UPDATES_PROXY_SERVER=

#Proxy Server Port
SOFTWARE_UPDATES_PROXY_PORT=

#Proxy Server Username
SOFTWARE_UPDATES_PROXY_USER=

#Proxy Server Password
SOFTWARE_UPDATES_PROXY_PASSWORD=

#The oracle home location. This can be an existing Oracle Home or a new Oracle Home
ORACLE_HOME=$Install_Base_Location/app/Oracle/Middleware/Oracle_Home

#Set this variable value to the Installation Type selected. e.g. BI Platform Distribution, BI Platform Distribution with Samples.
INSTALL_TYPE=BI Platform Distribution with Samples" > $Software_Location/oracle_analytics_server.rsp
}


# Oracle Analytics Server Installation
installOAS()
{
 echo "Installing OAS in Silent Mode....." 
 mkdir -p $Software_Location/oas 
 unzip $Software_Location/$OAS_FileName -d $Software_Location/oas
 sudo chown -R $User:$Group $Software_Location/oas
 java -jar $Software_Location/oas/Oracle_Analytics_Server_2024_Linux.jar -silent -responseFile $Software_Location/oracle_analytics_server.rsp -invPtrLoc $Install_Base_Location/app/oraInventory/oraInst.loc
 rm -rf $Software_Location/oas

 # Check if the bi Folder Exists
 if [[ -d "$Install_Base_Location/app/Oracle/Middleware/Oracle_Home/bi" ]]; then
	echo "bi folder exists, OAS installed successfully."
 else
	echo "Unable to install OAS."
	exit 1
 fi
}


# Clean the Files
cleanUP()
{
 echo "Cleaning up the files used in the Silent Installation....."
 rm -f $Software_Location/$JDK_FileName
 rm -f $Software_Location/$FMW_FileName
 rm -f $Software_Location/$OAS_FileName
 rm -f $Software_Location/fmw_12214_rsp_file.rsp
 rm -f $Software_Location/oracle_analytics_server.rsp
}


# Call the Functions
prerequisiteCheck
installJDK
installLibraries
createFMWRespone
installFMW
createOASResponse
installOAS
cleanUP