import subprocess
import json
import http.client
from datetime import datetime, timedelta
import argparse
import os
import csv

# REPLACE THE VALUES OF THE FOLLOWING VARIABLES
base64_encoded_clientid_clientsecret="<clientID:ClientSecret encoded string>"
domain="<Domain URL>"
username="<Username>"
password="<Password>"
scope="<Scope of OAC instance from Confidential Application>"
host = "<OAC URL>"
retention_number_of_days = 2  # Number of days to retain folders

# Calculate the cutoff date for retention
cutoff_date = datetime.now() - timedelta(days=retention_number_of_days)

def get_access_token():
    # Define the curl command
    curl_command = [
        "curl",
        "-H", f"Authorization: Basic {base64_encoded_clientid_clientsecret}",
        "-H", "Content-Type: application/x-www-form-urlencoded;charset=UTF-8",
        "--request", "POST",
        f"https://{domain}/oauth2/v1/token",
        "-d", f"grant_type=password&username={username}&password={password}&scope={scope}"
    ]

    # Run the curl command
    result = subprocess.run(curl_command, capture_output=True, text=True)

    # Define the output file name for the full response
    output_file = "oac_auth_token_json.txt"

    # Write the result to a file
    with open(output_file, "w") as file:
        file.write(result.stdout)

    print(f"Full response saved to {output_file}")

    if result.stderr:
        print(f"Errors encountered:\n{result.stderr}")

    # Parse the JSON response to extract the access_token
    try:
        response_data = json.loads(result.stdout)
        access_token = response_data.get("access_token")

        if access_token:
            token_file = "oac_auth_token.txt"
            with open(token_file, "w") as token_file:
                token_file.write(access_token)

            print(f"Access token saved to {token_file}")
        else:
            print("Access token not found in the response.")
            exit(1)

    except json.JSONDecodeError as e:
        print(f"Failed to parse JSON response: {e}")
        exit(1)

def copy_catalog_folder(source_id, dest_parent_id, token_file):
    # Read the bearer token from the text file
    if os.path.isfile(token_file):
        with open(token_file, 'r') as file:
            bearer_token = file.read().strip()
    else:
        print(f"Error: The token file '{token_file}' does not exist.")
        exit(1)

    # Define the API endpoint
    url = f"/api/20210901/catalog/folders/{source_id}/actions/copy"

    # Define the headers
    headers = {
        "Authorization": f"Bearer {bearer_token}",
        "Content-Type": "application/json"
    }

    current_datetime = datetime.now().strftime("%Y%m%d_%H%M%S")

    # Define the body of the request
    body = {
        "destParentId": dest_parent_id,
        "newItemName": f"AUTOBACKUP_{current_datetime}",
        "overwrite": True
    }

    # Convert the body to JSON format
    body_json = json.dumps(body)

    connection = http.client.HTTPSConnection(host)

    # Send the POST request
    connection.request("POST", url, body=body_json, headers=headers)

    # Get the response
    response = connection.getresponse()
    response_body = response.read().decode("utf-8")
    print("Status Code:", response.status)
    print("Response Body:", response_body)

    connection.close()

def fetch_catalog_folder_info(dest_parent_id, token_file, csv_filename="catalog_folder_info.csv"):
    # Read the bearer token from the text file
    if os.path.isfile(token_file):
        with open(token_file, 'r') as file:
            bearer_token = file.read().strip()
    else:
        print(f"Error: The token file '{token_file}' does not exist.")
        exit(1)

    # Initialize items list to get data from all pages
    all_items = []
    page = 1

    while True:
        # Define the API endpoint with pagination
        url = f"/api/20210901/catalog/folders/{dest_parent_id}?search=AUTOBACKUP&type=folders&sortBy=lastModified&sortOrder=DESC&page={page}"

        headers = {
            "Authorization": f"Bearer {bearer_token}"
        }

        connection = http.client.HTTPSConnection(host)

        # Send the GET request
        connection.request("GET", url, headers=headers)

        # Get the response
        response = connection.getresponse()
        response_body = response.read().decode("utf-8")

        # Extract the 'oa-page-count' field from response headers
        oa_page_count = response.getheader('oa-page-count')
        print(f"oa-page-count: {oa_page_count}")
        print("Status Code:", response.status)
        print("Response Body:", response_body)

        connection.close()

        # Parse the JSON response
        try:
            folder_data = json.loads(response_body)

            if isinstance(folder_data, list):
                items = folder_data
            elif isinstance(folder_data, dict):
                items = folder_data.get('items', [])
            else:
                print("Unexpected response format.")
                return

            all_items.extend(items)

            # Check if we have reached the last page
            if page >= int(oa_page_count):
                break
            page += 1

        except json.JSONDecodeError as e:
            print(f"Failed to parse JSON response: {e}")
            exit(1)

    # Write to CSV file
    with open(csv_filename, 'w', newline='') as csvfile:
        fieldnames = ['id', 'name', 'path', 'lastModified']
        writer = csv.DictWriter(csvfile, fieldnames=fieldnames)

        writer.writeheader()
        for item in all_items:
            last_modified = item.get('lastModified')
            if last_modified:
                last_modified_date = datetime.strptime(last_modified, "%Y-%m-%dT%H:%M:%SZ")
                if last_modified_date < cutoff_date:
                    writer.writerow({
                        'id': item.get('id'),
                        'name': item.get('name'),
                        'path': item.get('path'),
                        'lastModified': last_modified
                    })

    print(f"Catalog folder information saved to {csv_filename}")

def delete_catalog_folders(csv_filename, token_file):
    # Read the bearer token from the text file
    if os.path.isfile(token_file):
        with open(token_file, 'r') as file:
            bearer_token = file.read().strip()
    else:
        print(f"Error: The token file '{token_file}' does not exist.")
        exit(1)

    # Read the CSV file and delete the folders
    if os.path.isfile(csv_filename):
        with open(csv_filename, newline='') as csvfile:
            reader = csv.DictReader(csvfile)
            print(f"Deleting AUTOBACKUP folders created before {cutoff_date}")
            for row in reader:
                folder_id = row['id']
                delete_url = f"/api/20210901/catalog/folders/{folder_id}?recursive=true"

                headers = {
                    "Authorization": f"Bearer {bearer_token}"
                }

                connection = http.client.HTTPSConnection(host)

                # Send the DELETE request
                connection.request("DELETE", delete_url, headers=headers)

                # Get the response
                response = connection.getresponse()
                print(f"Deleting folder {folder_id}: Status Code: {response.status}")
                response_body = response.read().decode("utf-8")
                print("Response Body:", response_body)

                connection.close()
    else:
        print(f"Error: The CSV file '{csv_filename}' does not exist.")

# Main function
def main():
    # Get the access token
    get_access_token()

    # Set up argument parser for Catalog copy API
    parser = argparse.ArgumentParser(description='Call the OAC REST API to copy and delete catalog folders.')
    parser.add_argument('source_id', type=str, help='The source ID of the folder to be copied (e.g., L3NoYXJlZC9DYXRhbG9nQVBJ)')
    parser.add_argument('dest_parent_id', type=str, help='The destination parent ID where the folder will be copied (e.g., L3NoYXJlZC9DYXRhbG9nQVBJX0JrcA)')
    parser.add_argument('-t', '--token_file', type=str, default='oac_auth_token.txt', help='Path to the text file containing the bearer token (default: oac_auth_token.txt)')
    parser.add_argument('-c', '--csv', type=str, default='catalog_folder_info.csv', help='Path to the CSV file for storing catalog folder information (default: catalog_folder_info.csv)')

    # Parse the arguments
    args = parser.parse_args()

    # Copy the catalog folder using the obtained access token
    copy_catalog_folder(args.source_id, args.dest_parent_id, args.token_file)

    # Fetch catalog folders information and store it in a CSV file
    fetch_catalog_folder_info(args.dest_parent_id, args.token_file, args.csv)

    # Delete the folders before retention days
    delete_catalog_folders(args.csv, args.token_file)

if __name__ == "__main__":
    main()

