#!/bin/bash
#
# update_datasets.sh
# Version: 1.0.0
# Updated on: 2020/01/11
#
# Description: This script will update the datasets statistics when upgrading
# to Oracle Spatial Studio 20.1, using CURL command.
# If CURL command fails, you might remove http_proxy and https_proxy
# env vars, or set them to an appropriate value.
#
# Author: Daniel Cuadra
# Created on: 2020/01/11
#
if [[ $# -lt 2 ]]; then
  SCRIPT_NAME=$0
  SCRIPT_NAME=${SCRIPT_NAME##*/}
  echo
  echo "Missing arguments, use:"
  echo
  echo " ./${SCRIPT_NAME} {oauth_url} {full_oauth_token}"
  echo " For example: ./${SCRIPT_NAME} http://localhost:8080/spatialstudio/oauth/v1 eyJ0e...xwgg"
  exit 1
fi
# Script arguments
oauth_url=$1
oauth_token=$2
# Pre-init exit_status
exit_status=0
function remove_proxy {
  unset http_proxy
  unset https_proxy
}
function cleanup {
  echo
  if [[ $exit_status -ne 0 ]]; then
    echo "> FAILED"
  else
    echo "> DONE"
  fi
}
function test_connection {
  echo "  >> Testing repository connection ${oauth_url}/repository/status"
  # Retrieving repository status
  # Expected response:
  #
  #     {"metadataSchemaStatus":"READY"}
  #
  repository_status=$(curl -k -s "${oauth_url}/repository/status" -H "Authorization: Bearer ${oauth_token}"\
  | awk '{print $1}' | tr -s '\n' ' ' | tr -d ' {}:"' | tr '[:lower:]' '[:upper:]' | awk '{print substr($1,21)}' )
  exit_status=$?
  if [[ $exit_status -ne 0 ]]; then
    echo "  >> Connection failed"
    exit $exit_status
  fi
  if [[ $repository_status != 'READY' ]]; then
    exit_status=1
    echo "  >> Invalid repository status (expected READY)"
    echo "     - Check URL is valid and reachable"
    echo "     - Check OAuth Token is valid, non-expired, with read-write permissions"
    echo "     - Check your http_proxy and https_proxy env vars"
    echo "     - Check certificate is valid when using HTTPS"
    exit $exit_status
  else
    echo "  >> Connection succeeded"
  fi
}
function update_statistics {
  echo
  echo "> UPDATING DATASETS FROM ${oauth_url}/datasets?fields=id"
  # Retrieving datasets IDs
  # Expected response:
  #
  #     {"items":[
  #     {
  #       "id" : "2d8540b57f91a0a8d9826550d02567ef"
  #     },
  #     {
  #       "id" : "605bbeef1d7ebce21a4aa4ebffcf83eb"
  #     },
  #     ...]
  #     }
  #
  datasets_ids=$(curl -k -s "${oauth_url}/datasets?fields=id" -H "Authorization: Bearer $oauth_token"\
  | awk '{print $3}' | tr -d '"' | tr -s '\n' ' ')
  # Iterating over each dataset to update its statistics
  for ds_id in ${datasets_ids}
  do
    echo "  >> Updating Dataset statistics: ${ds_id}"
    # Updating statistics
    curl -k -s "$oauth_url/datasets/update_statistics/${ds_id}" -X POST -H "Authorization: Bearer ${oauth_token}" > /dev/null
  done
  echo "  >> All statistics were updated"
}
echo
echo
echo "****************************************"
echo "UPGRADING TO ORACLE SPATIAL STUDIO 20.1"
echo "****************************************"
echo
echo "> INITIALIZING"
trap cleanup EXIT
remove_proxy
test_connection
# Perform actual updates
update_statistics
